<?php

namespace App\Models;

use CodeIgniter\Model;
use App\Libraries\Auditoria;

class UserGroupModel extends Model
{
    // Nombre de la tabla de pivote
    protected $table         = 'admin.user_group';
    // Clave primaria compuesta
    protected $primaryKey    = ['user_id', 'group_id'];
    // Tipo de retorno de los resultados
    protected $returnType    = 'array';
    // No usamos timestamps para esta tabla de pivote
    protected $useTimestamps = false;

    // Campos permitidos para inserción
    protected $allowedFields = ['user_id', 'group_id'];

    /**
     * Asigna un usuario a un grupo específico.
     *
     * @param int $userId El ID del usuario.
     * @param int $groupId El ID del grupo.
     * @return bool True si la asignación fue exitosa, false en caso contrario.
     */
    public function assignUserToGroup(int $userId, int $groupId): bool
    {
        // Verifica si la asignación ya existe para evitar duplicados
        $exists = $this->where('user_id', $userId)
                       ->where('group_id', $groupId)
                       ->first();

        if ($exists) {
            return true; // Ya existe, no hacemos nada y retornamos éxito
        }

        // Inserta la nueva asignación
        return $this->insert(['user_id' => $userId, 'group_id' => $groupId]);
    }

    /**
     * Remueve un usuario de un grupo específico.
     *
     * @param int $userId El ID del usuario.
     * @param int $groupId El ID del grupo.
     * @return bool True si la remoción fue exitosa, false en caso contrario.
     */
    public function removeUserFromGroup(int $userId, int $groupId): bool
    {
        // Elimina la asignación específica
        return $this->where('user_id', $userId)
                    ->where('group_id', $groupId)
                    ->delete();
    }

    /**
     * Obtiene todos los grupos asignados a un usuario.
     *
     * @param int $userId El ID del usuario.
     * @return array Un array de objetos de grupo.
     */
    public function getGroupsForUser(int $userId): array
    {
        // Realiza un JOIN con la tabla admin.groups para obtener los detalles del grupo
        return $this->select('admin.groups.id, admin.groups.name, admin.groups.description')
                    ->join('admin.groups', 'admin.groups.id = admin.user_group.group_id')
                    ->where('admin.user_group.user_id', $userId)
                    ->findAll();
    }

    /**
     * Sincroniza los grupos de un usuario. Elimina los grupos no seleccionados
     * y añade los nuevos.
     *
     * @param int $userId El ID del usuario.
     * @param array $newGroupIds Un array de IDs de grupos que deben estar asignados al usuario.
     * @return void
     */
    public function syncUserGroups(int $userId, array $newGroupIds)
    {
        // Obtiene los IDs de los grupos actualmente asignados al usuario
        $currentGroups = $this->where('user_id', $userId)->findAll();
        $currentGroupIds = array_column($currentGroups, 'group_id');

        // Grupos a eliminar (están en currentGroups pero no en newGroupIds)
        $groupsToRemove = array_diff($currentGroupIds, $newGroupIds);
        foreach ($groupsToRemove as $groupId) {
            $this->removeUserFromGroup($userId, $groupId);
        }

        // Grupos a añadir (están en newGroupIds pero no en currentGroupIds)
        $groupsToAdd = array_diff($newGroupIds, $currentGroupIds);
        foreach ($groupsToAdd as $groupId) {
            $this->assignUserToGroup($userId, $groupId);
        }
    }
    public function insert_auditor($data)
    {
        $db = \Config\Database::connect();
        $db->transBegin();
        $auditoria = new Auditoria($db);
        $auditoria->actualizar_backend_pid();
        $this->insert($data);
        $db->transCommit();
        return true;
    } 
    
    public function update_auditor($id, $data)
    {
        $db = \Config\Database::connect();
        $db->transBegin();
        $auditoria = new Auditoria($db);
        $auditoria->actualizar_backend_pid();
        $this->update($id, $data);
        $db->transCommit();
        return true;
    } 
    
    public function delete_auditor($id)
    {
        $db = \Config\Database::connect();
        $db->transBegin();
        $auditoria = new Auditoria($db);
        $auditoria->actualizar_backend_pid();
        $this->delete($id);
        $db->transCommit();
        return true;
    }     
}
