<?php

namespace App\Controllers;

use App\Models\GroupModel;
use CodeIgniter\Exceptions\PageNotFoundException;

class Groups extends BaseController
{
    protected $model;

    public function __construct()
    {
        // Inicializa el modelo GroupModel
        $this->model = new GroupModel();
    }

    /**
     * Muestra la lista paginada de grupos con opciones de filtrado.
     */
    public function index()
    {
        // Define los filtros de búsqueda a partir de los parámetros GET
        $filtros = [
            'id'          => $this->request->getGet('id'),
            'name'        => $this->request->getGet('name'),
            'description' => $this->request->getGet('description'),
        ];

        // Define la cantidad de elementos por página
        $porPagina = 10;

        // Obtiene los items paginados aplicando los filtros y ordenando por 'id'
        $data['items'] = $this->model
            ->buscar($filtros) // Aplica los filtros definidos
            ->orderBy('id', 'ASC') // Ordena los resultados por 'id' de forma ascendente
            ->paginate($porPagina); // Pagina los resultados

        // Pasa el objeto pager a la vista para generar los enlaces de paginación
        $data['pager'] = $this->model->pager;
        // Pasa los filtros actuales a la vista para mantener los valores en el formulario de búsqueda
        $data['search'] = $filtros;

        // Carga la vista 'groups/index' con los datos preparados
        return view('groups/index', $data);
    }

    /**
     * Muestra los detalles de un grupo específico.
     *
     * @param int $id El ID del grupo a mostrar.
     */
    public function view($id)
    {
        // Busca un grupo por su ID
        $item = $this->model->find($id);

        // Si el grupo no se encuentra, lanza una excepción PageNotFoundException
        if (!$item) {
            throw PageNotFoundException::forPageNotFound("Registro de grupo no encontrado");
        }

        // Carga la vista 'groups/view' con los datos del grupo
        return view('groups/view', ['item' => $item]);
    }

    /**
     * Muestra el formulario para crear un nuevo grupo.
     */
    public function create()
    {
        // Carga la vista 'groups/create'
        return view('groups/create');
    }

    /**
     * Almacena un nuevo grupo en la base de datos.
     */
    public function store()
    {
        // Obtiene todos los datos enviados por POST
        $data = $this->request->getPost();
        // Inserta los datos en la tabla de grupos
        $this->model->insert_auditor($data);

        // Redirige a la página principal de grupos con un mensaje de éxito
        return redirect()->to('/groups')->with('mensaje', 'Grupo creado correctamente');
    }

    /**
     * Muestra el formulario para editar un grupo existente.
     *
     * @param int $id El ID del grupo a editar.
     */
    public function edit($id)
    {
        // Busca el grupo por su ID
        $item = $this->model->find($id);

        // Si el grupo no se encuentra, lanza una excepción PageNotFoundException
        if (!$item) {
            throw PageNotFoundException::forPageNotFound("Registro de grupo no encontrado");
        }

        // Carga la vista 'groups/edit' con los datos del grupo
        return view('groups/edit', ['item' => $item]);
    }

    /**
     * Actualiza un grupo existente en la base de datos.
     *
     * @param int $id El ID del grupo a actualizar.
     */
    public function update($id)
    {
        // Obtiene la instancia del servicio Request
        $request = \Config\Services::request();

        // Prepara los datos a actualizar a partir de los campos POST
        $data = [
            'name'        => $request->getPost('name'),
            'description' => $request->getPost('description'),
        ];

        // Limpia campos vacíos opcionales (los convierte en null)
        // Esto es útil para campos que pueden ser opcionales en la base de datos
        foreach ($data as $key => $value) {
            $data[$key] = (trim($value) === '') ? null : $value;
        }

        // Crea una nueva instancia del modelo (aunque ya está disponible en $this->model)
        // Se mantiene para consistencia con el código original del usuario, aunque $this->model->update($id, $data) sería suficiente.
        $model = new GroupModel();
        // Actualiza el registro en la base de datos
        $model->update_auditor($id, $data);

        // Redirige a la página principal de grupos con un mensaje de éxito
        return redirect()->to(base_url('groups'))->with('mensaje', 'Grupo actualizado correctamente');
    }

    /**
     * Elimina un grupo de la base de datos.
     *
     * @param int $id El ID del grupo a eliminar.
     */
    public function delete($id)
    {
        // Crea una nueva instancia del modelo (aunque ya está disponible en $this->model)
        $groupModel = new GroupModel();
        // Elimina el registro por su ID
        $groupModel->delete_auditor($id);

        // Redirige a la página principal de grupos con un mensaje de estado
        return redirect()->to(base_url('groups'))->with('status', 'Grupo eliminado correctamente');
    }
}
